#!/bin/bash
#: Title      : test_var4tac
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../var4tac$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f blood.fit ] || [ ! -f rbf.sif ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


if [ ! -f blood.bld ] || [ ! -f blood.dat ]; then
  rm -f renal.tac multi.tac
  fit2dat -c=0,420,1 blood.fit blood.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple -hdr=no -mid blood.dft blood.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=PMOD -hdr=no blood.dft blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f renal.tac ] ; then
  rm -f multi.tac
  b2t_h2o -nosub -fpt -voiname=cortex blood.bld 300 0.94 1.0 15 30 temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp.dat rbf.sif renal.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f multi.tac ] ; then
  rm -f multiw.tac
  tacadd --silent multi.tac renal.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent multi.tac multi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent multi.tac multi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent multi.tac multi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent multi.tac multi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent multi.tac multi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent multi.tac multi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent multi.tac multi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --silent multi.tac multi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f multiw.tac ] ; then
  cp -f multi.tac multiw.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacweigh --silent -wf multiw.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case: One BTAC in file with no frame durations. \n"
printf " Expected result 1: Noisy curve saved. \n\n"
printf " Expected result 2: AUC of noisy BTAC is close to AUC of original BTAC. \n\n"

rm -f noisy.dat

$PROGRAM blood.dat 5 noisy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="BTAC with and without noise" noisy.svg -l blood.dat noisy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -x=n -y=y -w=n -tacnames=n -xunit=n -yunit=n -abs=0.01 -rel=0.01 blood.dat noisy.dat
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

auc1=$(interpol -i -x=420 -header=no blood.dat)
printf "AUC := %s\n" $auc1 > auc1.ift
auc2=$(interpol -i -x=420 -header=no noisy.dat)
printf "AUC := %s\n" $auc2 > auc2.ift
cat auc1.ift auc2.ift
iftmatch -abs=200 auc1.ift auc2.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Same but data in PMOD BLD format. \n"
printf " Expected result 1: Noisy curve saved. \n\n"
printf " Expected result 2: Noise realisation is different from the previous one. \n\n"

rm -f noisy.bld

$PROGRAM blood.bld 5 noisy.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -mt="BTAC with and without noise" noisy_bld.svg -l blood.bld noisy.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -x=n -y=y -w=n -tacnames=n -xunit=n -yunit=n -abs=0.01 -rel=0.01 noisy.dat noisy.bld
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case: Lots of TACs in the file. \n"
printf " Expected result 1: Mean of noisy TACs is close to original. \n"
printf " Expected result 2: CV of noisy TACs is close to requested CV. \n\n"

rm -f output.tac mean.tac cv.tac

$PROGRAM multi.tac 10 output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmean -cv=cv.tac output.tac mean.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=1 -rel=20 renal.tac mean.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

dftinteg -avg cv.tac 10 350 cvmean.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift cvmean.res cvmean.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 cvmean.ift Mean 0.1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.0 \n"
printf " Test case: Option minsd. \n"
printf " Expected result 1: Mean of noisy TACs is still close to original. \n"
printf " Expected result 2: Peak CV of noisy TACs is still close to requested CV. \n"
printf " Expected result 3: Requested fraction of it is added also to the zero value. \n\n"

rm -f output2.tac mean2.tac cv2.tac sd2.tac

$PROGRAM -minsd=10 multi.tac 10 output2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmean -sd=sd2.tac -cv=cv2.tac output2.tac mean2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=1 -rel=20 renal.tac mean.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

taclist -ift cv2.tac > cv2.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.02 cv2.ift value[1][8] 0.10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

:: Max value 1.344304e+01
:: Its SD 1.344304e+00
:: and 10 percent of that is 1.344304e-01
taclist -ift sd2.tac > sd2.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.03 sd2.ift value[1][1] 1.344304e-01
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Weights \n"
printf "===================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case: Data file contains weights. \n"
printf " Expected result 1: Mean of noisy TACs is close to original. \n"
printf " Expected result 2: CVs of noisy TACs are close to correct. \n\n"

rm -f output.tac mean.tac cv.tac sd.tac

$PROGRAM multiw.tac 10 output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmean -cv=cv.tac output.tac mean.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=1 -rel=20 renal.tac mean.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacweigh -L multiw.tac > weights.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force cv.tac x weights.tac cvc.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
dftinteg -avg cvc.tac 10 350 cvmean.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift cvmean.res cvmean.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 cvmean.ift Mean 0.1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Error handling \n"
printf "===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption renal.tac 10 output.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM renal.tac 10 output.tac extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.2 \n"
printf " Test case: Missing argument(s). \n"
printf " Expected result: Error. \n\n"

$PROGRAM renal.tac 10  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: Invalid CV. \n"
printf " Expected result: Error. \n\n"

$PROGRAM renal.tac -1 output.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid CV% '-1'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM renal.tac CV output.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid CV% 'CV'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.1.1 \n"
printf " Test case: Small CV. \n"
printf " Expected result: Warning. \n\n"

$PROGRAM renal.tac 0.1 output.tac 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.2 \n"
printf " Test case: CV set to zero. \n"
printf " Expected result 1: Warning. \n"
printf " Expected result 2: Results saved without noise. \n\n"

rm -f output.tac

$PROGRAM renal.tac 0 output.tac 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt Warning "CV is set to 0%."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=0.001 renal.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.2.0 \n"
printf " Test case: Invalid minsd. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -minsd=999999 renal.tac 10 output.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid minimum SD percentage '-minsd=999999'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.3.0 \n"
printf " Test case: Non-existing input file. \n"
printf " Expected result: Error. \n\n"

rm -f output.tac
$PROGRAM nonexisting.tac 10 output.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "cannot open file (nonexisting.tac)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

