#!/bin/bash
#: Title      : test_tac2suv
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../tac2suv$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for  %s\n" $PROGRAM
printf "=====================================================================\n"

if [ ! -f dynamic.dat ]; then
  rm -f dynamic.dft dynamic.tac
  printf "1	0	5\n"   > dynamic.dat
  printf "3	3	20\n" >> dynamic.dat
  printf "5	9	44\n" >> dynamic.dat
  printf "7	11	43\n" >> dynamic.dat
  printf "10	10	32\n" >> dynamic.dat
  printf "15	8	19\n" >> dynamic.dat
  printf "20	8	12\n" >> dynamic.dat
  printf "25	5	8\n"  >> dynamic.dat
  printf "35	6	6\n"  >> dynamic.dat
  printf "45	5	3\n"  >> dynamic.dat
  printf "55	4	2\n"  >> dynamic.dat
fi

if [ ! -f dynamic.tac ]; then
  tacformat -f=PMOD -hdr=no dynamic.dat dynamic.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xset=min -yset=kBq/cc dynamic.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -yconv=Bq/cc dynamic.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f dynamic.dft ]; then
  tacformat -f=DFT -hdr=no dynamic.dat dynamic.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xset=min -yset=kBq/cc dynamic.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case 1: Plain format without units. \n"
printf " Test case 2: Calculating SUV. \n"
printf " Expected result 1: Correct output file. \n"
printf " Expected result 2: Warning on units. \n\n"

rm -f output.dat

printf "\n for comparison, calculate SUV TAC using taccalc (140/70=2)\n\n"
rm -f correct.dat
taccalc dynamic.dat div 2 correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM dynamic.dat 140 70 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt "Warning" "assuming unit kBq/mL"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output.dat unit "g/mL"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -rel=0.1 -y=y -yunit=n -xunit=n correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case 1: Plain format without units. \n"
printf " Test case 2: Calculating PID. \n"
printf " Expected result 1: Correct output file. \n"
printf " Expected result 2: Warning on units. \n\n"

rm -f output2.dat

printf "\n for comparison, calculate SUV TAC using taccalc (100%/200=0.5)\n\n"
rm -f correct2.dat
taccalc dynamic.dat x 0.5 correct2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM dynamic.dat 200 0 output2.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt "Warning" "assuming unit kBq/mL"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output2.dat unit "%ID/L"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -rel=0.1 -y=y -yunit=n -xunit=n correct2.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case 1: PMOD format with units in Bq/cc. \n"
printf " Test case 2: Calculating SUV. \n"
printf " Expected result: Correct output file. \n\n"

rm -f correct.tac output.tac stdout.txt

printf "\n for comparison, calculate SUV TAC using taccalc (1000*140/70=2000)\n\n"
taccalc dynamic.tac div 2000 correct.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM dynamic.tac 140 70 output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacunit output.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt unit "g/mL"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -rel=0.1 -y=y -yunit=n -xunit=y correct.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case: Output file name not given. \n"
printf " Expected result: Input file overwritten. \n\n"

cp -f dynamic.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM test.tac 140 70
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -yunit=y -xunit=y output.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.0 \n"
printf " Test case 1: DFT format with units. \n"
printf " Test case 2: Calculating SUV. \n"
printf " Expected result: Correct output file. \n\n"

rm -f correct.dft output.dft

printf "\n for comparison, calculate SUV TAC using taccalc (140/70=2)\n\n"
taccalc dynamic.dft div 2 correct.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM dynamic.dft 140 70 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacunit output.dft > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt unit "g/mL"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -rel=0.1 -y=y -yunit=n -xunit=y correct.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n----------------------------------------------------------------\n\n"

printf "\n 2.0.0 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM 140 70 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM dynamic.dat 140  output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM dynamic.dat 140 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: Too many arguments. \n"
printf " Expected result: Error. \n\n"

$PROGRAM dynamic.dat 140 70 output.dat extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "=====================================================================\n"
printf "  All passed!\n"
printf "=====================================================================\n"
exit 0

