#!/bin/bash
#: Title      : test_taccbv
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../taccbv$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


if [ ! -f blood.tac ] || [ ! -f blood0.dat ]; then
  printf "start[minutes]	end[kBq/mL]	Blood\n" > blood.tac
  printf "0	1	0\n"  >> blood.tac
  printf "1	2	40\n" >> blood.tac
  printf "2	4	20\n" >> blood.tac

  printf "0	0\n"    > blood0.dat
  printf "1	360\n" >> blood0.dat
  printf "2	40\n"  >> blood0.dat
  printf "3	20\n"  >> blood0.dat
  printf "3.9	20\n"  >> blood0.dat
fi

if [ ! -f tissue.dat ] || [ ! -f vb.dat ] || [ ! -f vb.csv ]; then
  printf "DFT         roi1     roi2\n"      > tissue.dat
  printf "exam0001    .        . \n"       >> tissue.dat
  printf "kBq/ml      .        . \n"       >> tissue.dat
  printf "Times (min) 0        0\n"        >> tissue.dat
  printf "0.00 1.00   0.00e+00 0.00e+00\n" >> tissue.dat
  printf "1.00 2.00   1.00e+02 1.00e+02\n" >> tissue.dat
  printf "2.00 4.00   5.00e+01 5.00e+01\n" >> tissue.dat

  printf "DFT         roi1     roi2\n"      > vb.dat
  printf "exam0002    .        . \n"       >> vb.dat
  printf "ml/ml       .        . \n"       >> vb.dat
  printf "Times (min) 0        0\n"        >> vb.dat
  printf "0.00 4.00   5.00e-02 2.50e-02\n" >> vb.dat

  printf "parameters,K1,K1/k2,k3,Vb\n"  > vb.csv
  printf "roi1,1.0,2.0,3.0,5.0\n"      >> vb.csv
  printf "roi2,1.0,2.0,3.0,2.5\n"      >> vb.csv
fi

if [ ! -f true_output1.dat ] || [ ! -f true_output2.dat ] || [ ! -f true_output3.dat ]; then
  printf "DFT1                 roi1        roi2       \n"  > true_output1.dat
  printf "exam0001             .           .          \n" >> true_output1.dat
  printf "kBq/ml               pl18        pl18       \n" >> true_output1.dat
  printf "Times (min)           0.000e+000  0.000e+000\n" >> true_output1.dat
  printf "    0.0000    1.0000 -9.000e+000 -9.000e+000\n" >> true_output1.dat
  printf "    1.0000    2.0000  9.000e+001  9.000e+001\n" >> true_output1.dat
  printf "    2.0000    4.0000  4.875e+001  4.875e+001\n" >> true_output1.dat

  printf "DFT1                 roi1        roi2       \n"  > true_output2.dat
  printf "exam0001             .           .          \n" >> true_output2.dat
  printf "kBq/ml               pl18        pl18       \n" >> true_output2.dat
  printf "Times (min)           0.000e+000  0.000e+000\n" >> true_output2.dat
  printf "    0.0000    1.0000 -9.474e+000 -9.474e+000\n" >> true_output2.dat
  printf "    1.0000    2.0000  9.474e+001  9.474e+001\n" >> true_output2.dat
  printf "    2.0000    4.0000  5.132e+001  5.132e+001\n" >> true_output2.dat

  printf "DFT1                 roi1        roi2       \n"  > true_output3.dat
  printf "exam0001             .           .          \n" >> true_output3.dat
  printf "kBq/ml               pl18        pl18       \n" >> true_output3.dat
  printf "Times (min)           0.000e+000  0.000e+000\n" >> true_output3.dat
  printf "    0.0000    1.0000  0.000e+000  0.000e+000\n" >> true_output3.dat
  printf "    1.0000    2.0000  9.000e+001  9.500e+001\n" >> true_output3.dat
  printf "    2.0000    4.0000  4.875e+001  4.938e+001\n" >> true_output3.dat
fi

if [ ! -f true_output4.dat ] || [ ! -f true_output5.dat ]; then
  printf "DFT1                 roi1        roi2       \n"  > true_output4.dat
  printf "exam0001             .           .          \n" >> true_output4.dat
  printf "kBq/ml               pl18        pl18       \n" >> true_output4.dat
  printf "Times (min)           0.0000e+00  0.0000e+00\n" >> true_output4.dat
  printf "    0.0000    1.0000  0.0000e+00  0.0000e+00\n" >> true_output4.dat
  printf "    1.0000    2.0000  1.0200e+02  1.0100e+02\n" >> true_output4.dat
  printf "    2.0000    4.0000  5.1000e+01  5.0500e+01\n" >> true_output4.dat

  printf "DFT1                 roi1        roi2       \n"  > true_output5.dat
  printf "exam0001             .           .          \n" >> true_output5.dat
  printf "kBq/ml               pl18        pl18       \n" >> true_output5.dat
  printf "Times (min)           0.0000e+00  0.0000e+00\n" >> true_output5.dat
  printf "    0.0000    1.0000  0.0000e+00  0.0000e+00\n" >> true_output5.dat
  printf "    1.0000    2.0000  9.7000e+01  9.8500e+01\n" >> true_output5.dat
  printf "    2.0000    4.0000  4.8500e+01  4.9250e+01\n" >> true_output5.dat
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case 1: Vb correction with fixed Vb. \n"
printf " Test case 2: Reversing previous correction with option -sim. \n"
printf " Expected result: Output TAC file with correct contents is made. \n\n"

rm -f output1.dat

$PROGRAM --force tissue.dat blood0.dat 5 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.0001 -x=y -y=y -xunit=y -yunit=y true_output1.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM --force -sim output1.dat blood0.dat 5 output1b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.0001 -x=y -y=y -xunit=y -yunit=y tissue.dat output1b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Same but without force option. \n"
printf " Expected result: Error because blood file contains no units. \n\n"

rm -f stdout.txt output1.dat

$PROGRAM tissue.dat blood0.dat 5 output1.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "incompatible units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.2 \n"
printf " Test case: Units are added. \n"
printf " Expected result: Works without option --force. \n\n"

cp -f blood0.dat temp.dat
rm -f output1.dat
tacunit -xset=min -yset=kBq/mL temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM tissue.dat temp.dat 5 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.0001 -x=y -y=y -xunit=y -yunit=y true_output1.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM --force -sim output1.dat blood0.dat 5 output1b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.0001 -x=y -y=y -xunit=y -yunit=y tissue.dat output1b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.3 \n"
printf " Test case: Units are added. Not the same but ones that can be converted. \n"
printf " Expected result: Works without option --force. \n\n"

tacunit -xset=min -yset=kBq/mL temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec -yconv=Bq/mL temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
rm -f output1.dat

$PROGRAM tissue.dat temp.dat 5 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.0001 -x=y -y=y -xunit=y -yunit=y true_output1.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM --force -sim output1.dat blood0.dat 5 output1b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.0001 -x=y -y=y -xunit=y -yunit=y tissue.dat output1b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case 1: Vb correction with fixed Vb, with option -tv. \n"
printf " Test case 2: Reversing previous correction with option -sim. \n"
printf " Expected result: Output TAC file with correct contents is made. \n\n"

rm -f output2.dat

$PROGRAM --force -tv tissue.dat blood0.dat 5 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.01 -x=y -y=y -xunit=y -yunit=y true_output2.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM --force -sim -tv output2.dat blood0.dat 5 output2b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.01 -x=y -y=y -xunit=y -yunit=y tissue.dat output2b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case: Same as before but Vb is given as fraction. \n"
printf " Expected result: Output TAC file with correct contents is made. \n\n"

rm -f output2.dat

$PROGRAM --force -tv tissue.dat blood0.dat 0.05 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.01 -x=y -y=y -xunit=y -yunit=y true_output2.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM --force -sim -tv output2.dat blood0.dat 0.05 output2b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.01 -x=y -y=y -xunit=y -yunit=y tissue.dat output2b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.0 \n"
printf " Test case: Vb correction based on Vb TAC file, with option -noneg. \n"
printf " Expected result: Output TAC file with correct contents is made. \n\n"

rm -f output3.dat

$PROGRAM --force -noneg tissue.dat blood0.dat vb.dat output3.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 -x=y -y=y -xunit=y -yunit=y true_output3.dat output3.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case 1: Vb simulation based on Vb TAC file, with option -sim and -pv. \n"
printf " Test case 2: Correcting it. \n"
printf " Expected result: Output TAC file with correct contents is made. \n\n"

rm -f output4.dat

$PROGRAM --verbose -pv -sim tissue.dat blood.tac vb.dat output4.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.01 -x=y -y=y -xunit=y -yunit=y true_output4.dat output4.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -pv output4.dat blood.tac vb.dat output4b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 -x=y -y=y -xunit=y -yunit=y tissue.dat output4b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.1 \n"
printf " Test case 1: Vb simulation based on Vb TAC file, with option -sim and -tv. \n"
printf " Test case 2: Correcting it. \n"
printf " Expected result: Output TAC file with correct contents is made. \n\n"

rm -f output5.dat

$PROGRAM --verbose -tv -sim tissue.dat blood.tac vb.dat output5.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.01 -x=y -y=y -xunit=y -yunit=y true_output5.dat output5.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -tv output5.dat blood.tac vb.dat output5b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 -x=y -y=y -xunit=y -yunit=y tissue.dat output5b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.0 \n"
printf " Test case 1: Vb simulation based on Vb PAR file, with option -sim and -pv. \n"
printf " Test case 2: Correcting it. \n"
printf " Test case 3: Vb is given as percentage. \n"
printf " Expected result: Output TAC file with correct contents is made. \n\n"

rm -f output4.dat

$PROGRAM --verbose -pv -sim tissue.dat blood.tac vb.csv output4.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.01 -x=y -y=y -xunit=y -yunit=y true_output4.dat output4.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -pv output4.dat blood.tac vb.csv output4b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 -x=y -y=y -xunit=y -yunit=y tissue.dat output4b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.1 \n"
printf " Test case 1: Vb simulation based on Vb PAR file, with option -sim and -tv. \n"
printf " Test case 2: Correcting it. \n"
printf " Test case 3: Vb is given as percentage. \n"
printf " Expected result: Output TAC file with correct contents is made. \n\n"

rm -f output5.dat

$PROGRAM --verbose -tv -sim tissue.dat blood.tac vb.csv output5.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.01 -x=y -y=y -xunit=y -yunit=y true_output5.dat output5.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -tv output5.dat blood.tac vb.csv output5b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 -x=y -y=y -xunit=y -yunit=y tissue.dat output5b.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.2 \n"
printf " Test case 1: TAC names in Vb PAR file do not match names in TTAC file. \n"
printf " Expected result: Correct output with --force, error without. \n\n"

cp -f tissue.dat temp.dat
tacren temp.dat 1 meh
tacren temp.dat 2 mmpfh
rm -f output4.dat

$PROGRAM --force -sim temp.dat blood.tac vb.csv output4.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.01 -x=y -y=y -xunit=y -yunit=y true_output4.dat output4.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "Vb file does not contain the same TACs."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM --verbose output4.dat blood.tac vb.csv output4b.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "Vb file does not contain the same TACs."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

