/** @file tacadd0.c
 *  @brief Add zero sample to the beginning of TAC data.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Adds a zero sample to the beginning of TAC data.",
  "If data samples already start from time (x) zero, then nothing is done.",
  " ",
  "Usage: @P [options] tacfile [outputfile]",
  " ",
  "Options:",
  " -x=<time>",
  "     Add late time point with zero value instead.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacsetx, tacsety, inpstart, tacframe, taccat, taccut",
  " ",
  "Keywords: TAC, time, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  char *cptr, tacfile[FILENAME_MAX], outfile[FILENAME_MAX];
  double latetime=nan("");
  TAC tac;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "X=", 2)==0) {
      if(atofCheck(cptr+2, &latetime)==0 && latetime>0.0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  if(ai<argc) strlcpy(tacfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(outfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) {
    fprintf(stderr, "Error: too many arguments.\n");
    return(1);
  }

  /* Is something missing? */
  if(!tacfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}
  if(!outfile[0]) strcpy(outfile, tacfile);
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("tacfile := %s\n", tacfile);
    printf("outfile := %s\n", outfile);
    if(!isnan(latetime)) printf("latetime := %g\n", latetime);
  }


  /*
   *  Read the file
   */
  if(verbose>1) printf("reading %s\n", tacfile);
  ret=tacRead(&tac, tacfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    printf("xunit := %s\n", unitName(tac.tunit));
    printf("yunit := %s\n", unitName(tac.cunit));
  }
  /* Sort the data by sample times (x) */
  ret=tacSortByTime(&tac, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(2);
  }


  /*
   *  Check whether zero or late sample exists already
   */
  if(isnan(latetime)) {
    if((tac.isframe && tac.x1[0]<=0.0) || (!tac.isframe && tac.x[0]<=0.0)) {
      fprintf(stderr, "Notice: zero sample exists; nothing to do.\n");
      tacFree(&tac); return(0);
    }
  } else {
    if((tac.isframe && tac.x2[tac.sampleNr-1]>=latetime) || (!tac.isframe && tac.x[tac.sampleNr-1]>=latetime)) {
      fprintf(stderr, "Notice: late sample exists; nothing to do.\n");
      tacFree(&tac); return(0);
    }
  }


  /*
   *  Determine the new sample x values
   */
  double newx, newx1, newx2; 
  newx=newx1=newx2=nan("");
  if(tac.isframe) {
    if(verbose>1) {
      int i; if(isnan(latetime)) i=0; else i=tac.sampleNr-1;
      printf("x1 := %g\n", tac.x1[i]);
      printf("x2 := %g\n", tac.x2[i]);
    }
    if(isnan(latetime) && tac.x1[0]>0.0) {
      newx2=tac.x1[0];
      newx1=newx2-(tac.x2[0]-tac.x1[0]);
      if(newx1<0.0) newx1=0.0;
      newx=0.5*(newx1+newx2);
    }
    if(!isnan(latetime)) {
      int i=tac.sampleNr-1;
      newx=latetime;
      newx1=tac.x2[i];
      newx2=latetime+(latetime-newx1);
    }
  } else {
    if(verbose>1) {
      int i; if(isnan(latetime)) i=0; else i=tac.sampleNr-1;
      printf("x[%d] := %g\n", i, tac.x[i]);
      if(tac.sampleNr>1) {
        if(isnan(latetime)) printf("x[1] := %g\n", tac.x[1]);
        else printf("x[%d] := %g\n", i-1, tac.x[i-1]);
      }
    }
    if(isnan(latetime) && tac.x[0]>0.0) {
      if(tac.sampleNr>1) {
        newx=tac.x[0]-(tac.x[1]-tac.x[0]);
        if(newx<0.0) newx=0.0;
      } else {
        newx=0.0;
      }
    }
    if(!isnan(latetime)) {
      newx=latetime;
    }
  }
  if(verbose>1) {
    if(tac.isframe) {
      printf("newx1 := %g\n", newx1);
      printf("newx2 := %g\n", newx2);
    } else {
      printf("newx := %g\n", newx);
    }
  }
  if(tac.isframe && (isnan(newx1) || isnan(newx2))) {
    fprintf(stderr, "Error: cannot add zero sample.\n");
    tacFree(&tac); return(3);
  }
  if(!tac.isframe && isnan(newx)) {
    fprintf(stderr, "Error: cannot add zero sample.\n");
    tacFree(&tac); return(3);
  }

  /*
   *  Add the zero sample
   */
  if(verbose>2) printf("adding zero sample\n");
  if(isnan(latetime)) {
    ret=tacAddZeroSample(&tac, &status);
  } else {
    ret=tacAllocateMoreSamples(&tac, 1);
  }
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: cannot add zero sample.\n");
    tacFree(&tac); return(4);
  }
  if(isnan(latetime)) {
    tac.x1[0]=newx1;
    tac.x2[0]=newx2;
    tac.x[0]=newx;
  } else {
    tac.sampleNr++;
    tac.x1[tac.sampleNr-1]=newx1;
    tac.x2[tac.sampleNr-1]=newx2;
    tac.x[tac.sampleNr-1]=newx;
    for(int i=0; i<tac.tacNr; i++) tac.c[i].y[tac.sampleNr-1]=0.0;
  }


  /*
   *  Save data 
   */
  if(verbose>1) printf("writing %s\n", outfile);
  FILE *fp; fp=fopen(outfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", outfile);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }
  if(verbose>0) printf("%s saved.\n", outfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
