/** @file tacdel.c
 *  @brief Delete certain TAC(s) inside TAC file.
 *  @details Application name was previously dftdel. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Delete specified TAC(s) inside a TAC file.",
  " ",
  "Usage: @P [options] file [tacname(s) or tacnumber(s)]",
  " ",
  "Options:",
  " -dry",
  "     Nothing is actually deleted, application only lists which TAC(s) would",
  "     be deleted.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: Delete TACs with name 'striatum' from file a123_12.dat",
  "     @P a123_06.dat striatum",  
  " ",
  "Example 2: Delete TAC numbers 3 and 4 from file a234.dat",
  "     @P a234.dat 3 4",  
  " ",
  "See also: taclist, tacadd, tacdelna, tacsort, tacunit, taccut, tacsplit",
  " ",
  "Keywords: TAC, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, i, n, dryMode=0;
  char *cptr, tacfile[FILENAME_MAX];
  TAC tac;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "DRY")==0) {
      dryMode=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* The first argument (non-option) is the filename */
  if(ai<argc) {
    strlcpy(tacfile, argv[ai], FILENAME_MAX); ai++;
  } else {
    fprintf(stderr, "Error: missing filename.\n");
    return(1);
  }
  if(ai>=argc) {
    fprintf(stderr, "Error: missing TAC name or number.\n");
    return(1);
  }
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile := %s\n", tacfile);
    printf("dryMode := %d\n", dryMode);
  }

  /* Read the file */
  if(verbose>1) printf("reading %s\n", tacfile);
  ret=tacRead(&tac, tacfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
  }

  /* Process the rest of the arguments: region names or numbers */
  for(; ai<argc; ai++) {
    n=tacSelectTACs(&tac, argv[ai], 0, &status);
    if(n<=0 && verbose>0)
      fprintf(stderr, "Warning: no match found for '%s'.\n", argv[ai]);
    else if(verbose>1) 
      printf("%d tac(s) match name '%s'\n", n, argv[ai]);
  }
  n=tacSelectedTACs(&tac);
  if(n<1) {
    fprintf(stderr, "Error: specified TACs not found in %s.\n", tacfile);
    tacFree(&tac); return(3);
  }
  if(verbose>1) printf("%d matching tac(s) found\n", n);
  if(n==tac.tacNr) {
    fprintf(stderr, "Warning: all TACs were selected for removal; aborted.\n");
    tacFree(&tac); return(3);
  }

  /*
   *  Delete the selected TACs
   */
  i=tac.tacNr-1; n=0; ret=TPCERROR_OK;
  while(i>=0) {
    if(tac.c[i].sw) {
      if(dryMode) {
        printf("marked for deletion: %s\n", tac.c[i].name); n++;
      } else {
        if(verbose>=0) printf("%s deleted\n", tac.c[i].name);
        ret=tacDeleteTACC(&tac, i); if(ret!=TPCERROR_OK) break; else n++;
      }
    }
    i--;
  }
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: cannot delete TAC.\n");
    tacFree(&tac); return(5);
  }
  if(dryMode) {
    printf("%d TAC(s) were marked to be deleted.\n", n);
    tacFree(&tac); return(0);
  } 

  /*
   *  Save data 
   */
  if(verbose>1) printf("writing %s\n", tacfile);
  FILE *fp; fp=fopen(tacfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }

  if(verbose>=0) printf("%d TAC(s) are deleted.\n", n);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
