/** @file tacinv.c
 *  @brief Invert TAC data (calculate 1/y).
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Inverts the y values (concentrations) in TAC file, that is, calculates 1/y.",
  " ",
  "Usage: @P [options] tacfile outputfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Values close to zero will be replaced by NaN.",
  " ",
  "See also: imginv, taccalc, fit_sigm, tacadd, taccuty",
  " ",
  "Keywords: TAC, modelling, simulation, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  char tacfile[FILENAME_MAX], outfile[FILENAME_MAX];
  TAC tac;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  for(; ai<argc; ai++) {
    if(!tacfile[0]) {
      strcpy(tacfile, argv[ai]); continue;
    } else if(!outfile[0]) {
      strcpy(outfile, argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!outfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("tacfile := %s\n", tacfile);
    printf("outfile := %s\n", outfile);
  }


  /*
   *  Read the file
   */
  if(verbose>1) printf("reading %s\n", tacfile);
  ret=tacRead(&tac, tacfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    printf("xunit := %s\n", unitName(tac.tunit));
    printf("yunit := %s\n", unitName(tac.cunit));
  }


  /*
   *  Invert the y data
   */
  int fi, ri, okNr=0, failNr=0;
  double f;
  for(ri=0; ri<tac.tacNr; ri++) {
    for(fi=0; fi<tac.sampleNr; fi++) {
      if(isnan(tac.c[ri].y[fi])) {failNr++; continue;}
      f=1.0/tac.c[ri].y[fi];
      if(isfinite(f)) {tac.c[ri].y[fi]=f; okNr++;} 
      else {tac.c[ri].y[fi]=nan(""); failNr++;}
      continue;
    }
  }
  if(verbose>1) {
    printf("  failNr := %d\n", failNr);
    printf("  okNr := %d\n", okNr);
  }
  if(okNr<1) {
    fprintf(stderr, "Error: invalid data contents.\n");
    tacFree(&tac); return(3);
  }


  /*
   *  Save data 
   */
  if(verbose>1) printf("writing %s\n", outfile);
  FILE *fp; fp=fopen(outfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", outfile);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }
  if(verbose>=0) printf("%s saved.\n", outfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
