/** @file tacmatch.c
 *  @brief Verify that two TAC files have matching contents; for SW testing.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Add decent tests, although library functions are tested.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Verify that the contents in two TAC files are similar.",
  "This can be used for automating software and analysis tool testing.",
  "By default exact match is required for numerical values, but this behavior",
  "can be changed with options -rel and -abs (below); if both are set, then",
  "values are considered to match is either of the requirements is met.",
  "Programs return code is 0, if files were matching, 10, if the files did",
  "not match, and 1-9 in case of error.",
  " ",
  "Usage: @P [options] filename1 filename2",
  " ",
  "Options:",
  " -x[=<Y|n>]",
  "     X values (sample times) are checked (y, default) or not checked (n).",
  " -y[=<Y|n>]",
  "     Y values (concentrations) are checked (y, default) or not checked (n).",
  " -w=<y|N>",
  "     Weights are checked (y) or not checked (n, default).",
  " -tacnames[=<y|N>]",
  "     TAC names are checked (y) or not checked (n, default).",
  " -xunit[=<y|N>]",
  "     Time unit is checked (y) or not checked (n, default).",
  " -yunit[=<y|N>]",
  "     Concentration unit is checked (y) or not checked (n, default).",
  " -abs=<value>",
  "     Absolute differences must not exceed the specified limit.",
  " -rel=<value>",
  "     Relative differences must not exceed the specified percent limit.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1:",
  "     @P data1.tac data2.tac",
  "Example 2:",
  "     @P -abs=0.02 -rel=5 data1.dat data2.dat",
  " ",
  "See also: taclist, iftmatch, imgmatch",
  " ",
  "Keywords: TAC, tool, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, matchNr=0;
  char *cptr, tacfile1[FILENAME_MAX], tacfile2[FILENAME_MAX];
  TAC tac1, tac2;
  double test_abs=-1.0;
  double test_rel=-1.0;
  /* Optional tests */
  int xvals=1;
  int yvals=1;
  int wvals=0;
  int tacnames=0;
  int xunit=0;
  int yunit=0;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac1); tacInit(&tac2);
  tacfile1[0]=tacfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "ABS=", 4)==0) {
      test_abs=atofVerified(cptr+4); if(!isnan(test_abs)) continue;
    } else if(strncasecmp(cptr, "REL=", 4)==0) {
      test_rel=atofVerified(cptr+4); if(!isnan(test_rel)) continue;
    } else if(strcasecmp(cptr, "X")==0) {
      xvals=1; continue;
    } else if(strncasecmp(cptr, "X=", 2)==0) {
      cptr+=2;
      if(strncasecmp(cptr, "YES", 1)==0) {xvals=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {xvals=0; continue;}
    } else if(strcasecmp(cptr, "Y")==0) {
      yvals=1; continue;
    } else if(strncasecmp(cptr, "Y=", 2)==0) {
      cptr+=2;
      if(strncasecmp(cptr, "YES", 1)==0) {yvals=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {yvals=0; continue;}
    } else if(strncasecmp(cptr, "W=", 2)==0) {
      cptr+=2;
      if(strncasecmp(cptr, "YES", 1)==0) {wvals=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {wvals=0; continue;}
    } else if(strcasecmp(cptr, "XUNIT")==0) {
      xunit=1; continue;
    } else if(strncasecmp(cptr, "XUNIT=", 6)==0) {
      cptr+=6;
      if(strncasecmp(cptr, "YES", 1)==0) {xunit=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {xunit=0; continue;}
    } else if(strcasecmp(cptr, "YUNIT")==0) {
      yunit=1; continue;
    } else if(strncasecmp(cptr, "YUNIT=", 6)==0) {
      cptr+=6;
      if(strncasecmp(cptr, "YES", 1)==0) {yunit=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {yunit=0; continue;}
    } else if(strcasecmp(cptr, "TACNAMES")==0) {
      tacnames=1; continue;
    } else if(strncasecmp(cptr, "TACNAMES=", 9)==0) {
      cptr+=9;
      if(strncasecmp(cptr, "YES", 1)==0) {tacnames=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {tacnames=0; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  for(; ai<argc; ai++) {
    if(!tacfile1[0]) {
      strcpy(tacfile1, argv[ai]); continue;
    } else if(!tacfile2[0]) {
      strcpy(tacfile2, argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!tacfile2[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("tacfile1 := %s\n", tacfile1);
    printf("tacfile2 := %s\n", tacfile2);
    if(test_abs>=0.0) printf("test_abs := %g\n", test_abs);
    if(test_rel>=0.0) printf("test_rel := %g\n", test_rel);
    printf("xvals := %d\n", xvals);
    printf("yvals := %d\n", yvals);
    printf("wvals := %d\n", wvals);
    printf("xunit := %d\n", xunit);
    printf("yunit := %d\n", yunit);
    printf("tacnames := %d\n", tacnames);
    fflush(stdout);
  }

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /*
   *  Read TAC data
   */
  if(verbose>1) printf("\nreading %s\n", tacfile1);
  ret=tacRead(&tac1, tacfile1, &status);
  if(ret) {
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), tacfile1);
    tacFree(&tac1); tacFree(&tac2);
    return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac1.format));
    printf("tacNr := %d\n", tac1.tacNr);
    printf("sampleNr := %d\n", tac1.sampleNr);
    if(tac1.isframe) printf("frames := yes\n"); else printf("frames := no\n");
    fflush(stdout);
  }
  if(verbose>1) printf("\nreading %s\n", tacfile2);
  ret=tacRead(&tac2, tacfile2, &status);
  if(ret) {
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), tacfile2);
    tacFree(&tac1); tacFree(&tac2);
    return(3);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac2.format));
    printf("tacNr := %d\n", tac2.tacNr);
    printf("sampleNr := %d\n", tac2.sampleNr);
    if(tac2.isframe) printf("frames := yes\n"); else printf("frames := no\n");
    printf("\n");
    fflush(stdout);
  }


  /*
   *  Check units
   */
  if(xunit || yunit) {
    if(verbose>0) printf("test units\n");
    ret=tacCompareUnit(&tac1, &tac2, &status);
    if(yunit && (ret==1 || ret==3)) {
      fprintf(stdout, "  concentration units in %s and %s do not match.\n",
        tacfile1, tacfile2);
      tacFree(&tac1); tacFree(&tac2);
      return(10);
    }
    if(xunit && (ret==2 || ret==3)) {
      fprintf(stdout, "  time units in %s and %s do not match.\n",
        tacfile1, tacfile2);
      tacFree(&tac1); tacFree(&tac2);
      return(10);
    }
    matchNr++;
  }  

  /*
   *  Check x values
   */
  if(xvals) {
    /* If xunit is not tested, convert units if necessary and possible */
    if(!xunit && tac1.tunit!=tac2.tunit) {
      ret=tacXUnitConvert(&tac2, tac1.tunit, &status);
      if(verbose>1) {
        if(ret==TPCERROR_OK) printf("  sample time units converted.\n");
        else printf("  sample time units could not be converted.\n");
      }
    }
    if(verbose>0) printf("test sample times\n");
    if(tacCompareTimes(&tac1, &tac2, test_abs, 0.01*test_rel, &status)) {
      fprintf(stdout, "  time samples in %s and %s do not match.\n",
        tacfile1, tacfile2);
      tacFree(&tac1); tacFree(&tac2);
      return(10);
    }
    matchNr++;
  }
  
  /*
   *  Check y values
   */
  if(yvals) {
    /* If yunit is not tested, convert units if necessary and possible */
    if(!yunit && tac1.cunit!=tac2.cunit) {
      ret=tacYUnitConvert(&tac2, tac1.cunit, &status);
      if(verbose>1) {
        if(ret==TPCERROR_OK) printf("  concentration units converted.\n");
        else printf("  concentration units could not be converted.\n");
      }
    }
    if(verbose>0) printf("test sample concentrations\n");
    if(tacCompareConc(&tac1, &tac2, -1, test_abs, 0.01*test_rel, &status)) {
      fprintf(stdout, "  concentrations in %s and %s do not match.\n",
        tacfile1, tacfile2);
      tacFree(&tac1); tacFree(&tac2);
      return(10);
    }
    matchNr++;
  }
  
  /*
   *  Check weights
   */
  if(wvals) {
    if(verbose>0) printf("test sample weights\n");
    if(tacCompareWeights(&tac1, &tac2, test_abs, 0.01*test_rel, &status)) {
      fprintf(stdout, "  weights in %s and %s do not match.\n",
        tacfile1, tacfile2);
      tacFree(&tac1); tacFree(&tac2);
      return(10);
    }
    matchNr++;
  }

  /*
   *  Check TAC names
   */
  if(tacnames) {
    if(verbose>0) printf("test tac names\n");
    if(tacCompareNames(&tac1, &tac2, -1, &status)!=0) {
      fprintf(stdout, "  TAC names in %s and %s do not match.\n",
        tacfile1, tacfile2);
      tacFree(&tac1); tacFree(&tac2);
      return(10);
    }
    matchNr++;
  }

  tacFree(&tac1); tacFree(&tac2);

  if(matchNr==0) {
    fprintf(stderr, "Error: no tests applied.\n");
    return(1);
  } else if(verbose>=0) 
    fprintf(stdout, "Match was found.\n");

  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
