/** @file tacrange.c
 *  @brief List tac time range.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the time range in TAC datafile(s) in DFT, CSV, PMOD, or SIF formats.",
  " ",
  "Usage: @P [options] filename(s)",
  " ",
  "Options:",
  " -unit=<sec|min>",
  "     Times are shown in seconds, or minutes (default)",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacframe, tactime, dftmax, inpstart, tacunit, tacsetx",
  " ",
  "Keywords: TAC, SIF, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    ret, fileNr=0, ffi=0;
  int    tunit=UNIT_MIN;
  char  *cptr, tacfile[FILENAME_MAX];
  TAC    tac;
  double xmin, xmax;
  double min_xmin, max_xmax, max_xmin, min_xmax;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "UNIT=", 5)==0) {
      tunit=unitIdentify(cptr+5); if(unitIsTime(tunit)) continue;
    } else if(strncasecmp(cptr, "TUNIT=", 6)==0) {
      tunit=unitIdentify(cptr+6); if(unitIsTime(tunit)) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  fileNr=0;
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai;
    fileNr++;
  }

  /* Check that we got what we need */
  if(fileNr<1) {tpcPrintUsage(argv[0], info, stderr); tacFree(&tac); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("required_timeunit := %s\n", unitName(tunit));
    printf("fileNr := %d\n", fileNr);
  }


  /*
   *  Process each file
   */
  fileNr=0;
  for(ai=ffi; ai<argc; ai++) {
  
    /* Read TAC file */
    strcpy(tacfile, argv[ai]);
    fprintf(stdout, "%s : \n", tacfile); fflush(stdout);
    ret=tacRead(&tac, tacfile, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(2);
    }
    if(verbose>1) {
      printf("fileformat := %s\n", tacFormattxt(tac.format));
      printf("tacNr := %d\n", tac.tacNr);
      printf("sampleNr := %d\n", tac.sampleNr);
      printf("original_timeunit := %s\n", unitName(tac.tunit));
    }

    /* Convert time unit */
    ret=tacXUnitConvert(&tac, tunit, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Warning: cannot convert time unit from %s to %s\n", 
        unitName(tac.tunit), unitName(tunit));
    }

    /* Get the time range */
    ret=tacXRange(&tac, &xmin, &xmax);
    if(ret!=0) {
      fprintf(stderr, "Warning: cannot find time range in %s\n", tacfile);
      if(verbose>1) printf("ret := %d\n", ret);
      tacFree(&tac);
      continue;
    }

    /* Print time range */
    fprintf(stdout, "time_min := %g\n", xmin);
    fprintf(stdout, "time_max := %g\n", xmax);
    fflush(stdout);

    /* Set min and max start time so far */
    if(fileNr==0 || xmin<min_xmin) min_xmin=xmin;
    if(fileNr==0 || xmin>max_xmin) max_xmin=xmin;
    /* Set min and max end time so far */
    if(fileNr==0 || xmax<min_xmax) min_xmax=xmax;
    if(fileNr==0 || xmax>max_xmax) max_xmax=xmax;

    tacFree(&tac);
    fileNr++;
  } // next TAC

  if(fileNr==0) {
    fprintf(stderr, "Error: no min/max could be determined.\n");
    return(3);
  }

  /*
   *  Print the min and max of all files
   */
  if(fileNr>1) {
    fprintf(stdout, "all_files :\n");
    if(verbose>0) fprintf(stdout, "timeunit := %s\n", unitName(tunit));
    fprintf(stdout, "min_time_min := %g\n", min_xmin);
    fprintf(stdout, "max_time_max := %g\n", max_xmax);
    fprintf(stdout, "max_time_min := %g\n", max_xmin);
    fprintf(stdout, "min_time_max := %g\n", min_xmax);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
