/** @file tacsetx.c
 *  @brief Replace TAC x column (usually time) with a y column from another TAC file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Adds specified TAC from file2 as x (time) column to file1, or if not given,",
  "then copy sample times from file2.",
  "Files must have the same number of samples (time frames).",
  "If file1 contains only one data column, that is assumed to represent y",
  "values, and x values are added as new column to the left.",
  " ",
  "Usage: @P [Options] tacfile1 tacfile2 [tacid]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacmultx, tacsety, taclist, tacadd0, tacframe, addtimes, tacformat",
  " ",
  "Keywords: TAC, tool, time, simulation",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   ret, voi=-1;
  char  tacfile1[FILENAME_MAX], tacfile2[FILENAME_MAX], rname[MAX_TACNAME_LEN];
  TAC   tac1, tac2;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac1); tacInit(&tac2);
  tacfile1[0]=tacfile2[0]=rname[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //   char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  if(ai<argc) {strlcpy(tacfile1, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(tacfile2, argv[ai++], FILENAME_MAX);}
  for(int n=0; ai<argc; ai++, n++) { // collect TAC id optionally from multiple parts
    if(!roinameAddField(rname, argv[ai], n, MAX_TACNAME_LEN)) {
      fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
      return(1);
    }
  }

  /* Is something missing? */
  if(!tacfile2[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("tacfile1 := %s\n", tacfile1);
    printf("tacfile2 := %s\n", tacfile2);
    if(rname[0]) printf("rname := %s\n", rname);
    fflush(stdout);
  }


  /*
   *  Read the files
   */
  if(verbose>1) printf("reading %s\n", tacfile1);
  ret=tacRead(&tac1, tacfile1, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac1); return(2);
  }
  if(verbose>2) {
    printf("fileformat1 := %s\n", tacFormattxt(tac1.format));
    printf("tacNr1 := %d\n", tac1.tacNr);
    printf("sampleNr1 := %d\n", tac1.sampleNr);
    printf("xunit1 := %s\n", unitName(tac1.tunit));
    printf("yunit1 := %s\n", unitName(tac1.cunit));
    printf("isframe1 := %d\n", tac1.isframe);
    fflush(stdout);
  }
  if(verbose>1) printf("reading %s\n", tacfile2);
  ret=tacRead(&tac2, tacfile2, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac1); tacFree(&tac2); return(3);
  }
  if(verbose>2) {
    printf("fileformat2 := %s\n", tacFormattxt(tac2.format));
    printf("tacNr2 := %d\n", tac2.tacNr);
    printf("sampleNr2 := %d\n", tac2.sampleNr);
    printf("xunit2 := %s\n", unitName(tac2.tunit));
    printf("yunit2 := %s\n", unitName(tac2.cunit));
    printf("isframe2 := %d\n", tac2.isframe);
    fflush(stdout);
  }
  /* Check if file actually contained only y values and no x values at all */
  if(tac1.tacNr<=1 && tac1.isframe==0 && tacYNaNs(&tac1, 0)==tac1.sampleNr) {
    if(verbose>0) {printf("Note: initially no x column.\n"); fflush(stdout);}
    /* Move values from x column to the first y column */
    for(int i=0; i<tac1.sampleNr; i++) {
      tac1.c[0].y[i]=tac1.x[i];
      tac1.x[i]=nan("");
    }
  }

  /* Check sample number */
  if(tac1.sampleNr!=tac2.sampleNr) {
    fprintf(stderr, "Error: datafiles have different sample nr.\n");
    tacFree(&tac1); tacFree(&tac2); return(4);
  }

  if(rname[0]) {
    /* Select the region to be copied as x */
    if(verbose>1) printf("selecting TAC to be copied as x.\n");
    int n=tacSelectTACs(&tac2, rname, 0, &status);
    if(n<=0) {
      fprintf(stderr, "Error: no match found for '%s'.\n", rname);
      tacFree(&tac1); tacFree(&tac2); return(5);
    }
    if(n>1) {
      fprintf(stderr, "Error: more than one TAC matches '%s'.\n", rname);
      tacFree(&tac1); tacFree(&tac2); return(5);
    }
    for(int i=0; i<tac2.tacNr; i++) if(tac2.c[i].sw) {voi=i; break;}
  }


  if(rname[0]) {
    /* Add specified TAC as x */
    if(verbose>1) printf("adding region '%s' as x\n", tac1.c[voi].name);
    for(int i=0; i<tac1.sampleNr; i++)
      tac1.x[i]=tac1.x1[i]=tac1.x2[i]=tac2.c[voi].y[i];
    /* Make sure that times (x) are specified as single column only */
    tac1.isframe=0;
  } else {
    /* Copy x from the other file */
    if(verbose>1) printf("adding x from '%s'\n", tacfile2);
    for(int i=0; i<tac1.sampleNr; i++) {
      tac1.x[i]=tac2.x[i]; tac1.x1[i]=tac2.x1[i]; tac1.x2[i]=tac2.x2[i];
    }
    tac1.isframe=tac2.isframe;
    if(unitIsTime(tac2.tunit)) tac1.tunit=tac2.tunit;
  }
  tacFree(&tac2);


  /*
   *  Save data 
   */
  if(verbose>2) printf("writing %s\n", tacfile1);
  FILE *fp; fp=fopen(tacfile1, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile1);
    tacFree(&tac1); return(11);
  }
  ret=tacWrite(&tac1, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac1);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond

