/** @file tacsety.c
 *  @brief Replace specified y (concentration) values with given value
 *  for SW testing.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Replace specified TAC y (concentration) value(s) with given value.",
  "If file does not exist, then a new file is created.",
  " ",
  "Usage: @P [Options] tacfile tac frame value",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Tac and frame must be set to the sequence number of the TAC and frame;",
  "set tac and/or frame to 0 to replace value in all TACs and/or frames.",
  "To simulate missing value(s), set value to 'nan'.",
  " ",
  "See also: tacsetx, tacmultx, taclist, tacadd, taclkup, tacunit, taccuty",
  " ",
  "Keywords: TAC, tool, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    ret, row=-1, col=-1;
  double v=nan("");
//char  *cptr;
  char   tacfile[FILENAME_MAX];
  TAC    tac;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac); tacfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  ret=0;
  for(; ai<argc; ai++) {
    if(!tacfile[0]) {
      strlcpy(tacfile, argv[ai], FILENAME_MAX); continue;
    } else if(col<0) {
      col=atoi(argv[ai]); if(col>=0) continue;
    } else if(row<0) {
      row=atoi(argv[ai]); if(row>=0) continue;
    } else if(ret==0) {
      if(strncasecmp(argv[ai], "NAN", 2)==0) v=nan("");
      else v=atofVerified(argv[ai]);
      ret++; continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }
  /* Is something missing? */
  if(ret==0) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("tacfile := %s\n", tacfile);
    printf("row := %d\n", row);
    printf("col := %d\n", col);
    printf("value := %g\n", v);
  }

  /*
   *  If TAC file does not exist, then create TAC data.
   *  If it exists, then read it.
   */
  if(access(tacfile, 0) == -1) {
    if(verbose>1) printf("making TAC data\n");
    /* If TAC file is to be created, we must know the row and col nr */
    if(row<1 || col<1) {
      fprintf(stderr, "Error: invalid TAC or frame.\n");
      tacFree(&tac); return(2);
    }
    ret=tacAllocate(&tac, row, col);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(3);
    }
    tac.tacNr=col; tac.sampleNr=row;
    row=col=0; // all values are filled with the value
    tac.format=TAC_FORMAT_PMOD;
  } else {
    if(verbose>1) printf("reading %s\n", tacfile);
    ret=tacRead(&tac, tacfile, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(2);
    }
    if(verbose>2) {
      printf("fileformat := %s\n", tacFormattxt(tac.format));
      printf("tacNr := %d\n", tac.tacNr);
      printf("sampleNr := %d\n", tac.sampleNr);
      printf("xunit := %s\n", unitName(tac.tunit));
      printf("yunit := %s\n", unitName(tac.cunit));
    }
    /* Check sample number */
    if(tac.sampleNr<row) {
      fprintf(stderr, "Error: TAC file does not contain frame %d.\n", row);
      tacFree(&tac); return(3);
    }
    /* Check tac number */
    if(tac.tacNr<col) {
      fprintf(stderr, "Error: TAC file does not contain tac %d.\n", col);
      tacFree(&tac); return(4);
    }
  }

  /*
   *  Set the y value
   */
  for(int i=0; i<tac.tacNr; i++) {
    if(col!=0 && col!=i+1) continue;
    for(int j=0; j<tac.sampleNr; j++) {
      if(row==0 || row==j+1) tac.c[i].y[j]=v;
    }
  }


  /*
   *  Save TAC data 
   */
  if(verbose>2) printf("writing %s\n", tacfile);
  FILE *fp; fp=fopen(tacfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond

