/** @file tacsplit.c
 *  @brief Split TAC file into separate TAC files with one TAC in each.
 *  @details Application name was previously dftbreak. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Breaks up a TAC file into separate files, each containing one TAC.",
  "New files will be written in the current working directory by default.",
  " ",
  "Usage: @P [options] filename",
  " ",
  "Options:",
  " -name=<filename_tacid | filename_nr | tacid | nr>",
  "     Set the naming of output files. Options are the name of original",
  "     filename followed by TAC name (default) or number, or just TAC name",
  "     or TAC number.",
  " --force",
  "     Splitted TACs are saved even when they will overwrite existing files",
  "     or do not contain any data.",
  " -keeppath",
  "     Splitted files are saved in the same folder as the original file.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P -name=filename_tacid iea345.tac",
  " ",
  "See also: taclist, tacadd, taccut, tacformat, tacren, tacdel, tacsort",
  " ",
  "Keywords: TAC, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   ret;
  /** 0=filename_tacid, 1=filename_nr, 2=tacid, 3=nr */
  int   nameType=0;
  int   force=0;
  int   keeppath=0;
  char *cptr, tacfile[FILENAME_MAX], outfile[FILENAME_MAX];
  TAC   tac;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "FORCE")==0) {
      force=1; continue;
    } else if(strncasecmp(cptr, "KEEPPATH", 2)==0) {
      keeppath=1; continue;
    } else if(strncasecmp(cptr, "NAME=", 5)==0) {
      cptr+=5;
      if(strcasecmp(cptr, "filename_tacid")==0) {nameType=0; continue;}
      if(strcasecmp(cptr, "filename_nr")==0) {nameType=1; continue;}
      if(strcasecmp(cptr, "tacid")==0) {nameType=2; continue;}
      if(strcasecmp(cptr, "nr")==0) {nameType=3; continue;}
      if(strcasecmp(cptr, "fn_tacid")==0) {nameType=0; continue;}
      if(strcasecmp(cptr, "fn_nr")==0) {nameType=1; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Arguments */
  if(ai<argc) {strlcpy(tacfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]); return(1);}

  /* Is something missing? */
  if(!tacfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("tacfile := %s\n", tacfile);
    printf("nameType := %d\n", nameType);
    printf("force := %d\n", force);
    printf("keeppath := %d\n", keeppath);
  }


  /*
   *  Read the file
   */
  if(verbose>1) printf("reading %s\n", tacfile);
  ret=tacRead(&tac, tacfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    printf("xunit := %s\n", unitName(tac.tunit));
    printf("yunit := %s\n", unitName(tac.cunit));
  }
  /* Check that there are more than one TAC */
  if(tac.tacNr<2) {
    fprintf(stderr, "Error: %s contains only one TAC.\n", tacfile);
    tacFree(&tac); return(3);
  }


  /*
   *  If TAC names are to be used, then check that names are all individual
   */
  if(nameType==0 || nameType==2) {
    if(!tacIndividualNames(&tac)) {
      fprintf(stderr, "Error: %s does not contain individual TAC names.\n", tacfile);
      tacFree(&tac); return(4);
    }
  }

  /*
   *  Check that none of the new files will not overwrite anything
   *  unless user wants to overwrite files.
   */
  if(force==0) for(int i=0; i<tac.tacNr; i++) {
    /* Make the new filename for this TAC */
    outfile[0]=(char)0;
    /* If original filename is to included, then add it */
    if(nameType==0 || nameType==1) {
      strcat(outfile, tacfile);
      if(keeppath==0) filenameRmPath(outfile); 
      filenameRmExtensions(outfile);
      strcat(outfile, "_");
    } else if(keeppath!=0) { // or just the path, if requested
      strcpy(outfile, tacfile);
      char *cptr=strrchr(outfile, '/'); if(cptr==NULL) cptr=strrchr(outfile, '\\');
      if(cptr!=NULL) {cptr++; *cptr=(char)0;}
    }
    /* Add either TAC name or number */
    if(nameType==0 || nameType==2) {
      strncat(outfile, tac.c[i].name, FILENAME_MAX-strlen(outfile)-1);
    } else {
      strncatIntZ(outfile, 1+i, tac.tacNr, FILENAME_MAX-strlen(outfile)-1);
    }
    /* Add the original extension */
    strncat(outfile, filenameGetExtensions(tacfile), FILENAME_MAX-strlen(outfile)-1);
    if(verbose>2) printf("  %s\n", outfile);
    /* Check that file does not exist */
    if(access(outfile, 0)==-1) continue;
    fprintf(stderr, "Error: %s exists; use option -force to overwrite.\n", outfile);
    return(10);
  }

  /*
   *  Save each TAC
   */
  FILE *fp;
  int origTacNr=tac.tacNr;
  tac.tacNr=1;
  for(int i=0; i<origTacNr; i++) {
    /* Make the new filename for this TAC */
    outfile[0]=(char)0;
    /* If original filename is to included, then add it */
    if(nameType==0 || nameType==1) {
      strcat(outfile, tacfile);
      if(keeppath==0) filenameRmPath(outfile); 
      filenameRmExtensions(outfile);
      strcat(outfile, "_");
    } else if(keeppath!=0) { // or just the path, if requested
      strcpy(outfile, tacfile);
      char *cptr;
      cptr=strrchr(outfile, '/'); if(cptr==NULL) cptr=strrchr(outfile, '\\');
      if(cptr!=NULL) {cptr++; *cptr=(char)0;}
    }
    /* Add either TAC name or number */
    if(nameType==0 || nameType==2) {
      strncat(outfile, tac.c[i].name, FILENAME_MAX-strlen(outfile)-1);
    } else {
      strncatIntZ(outfile, 1+i, tac.tacNr, FILENAME_MAX-strlen(outfile)-1);
    }
    /* Add the original extension */
    strncat(outfile, filenameGetExtensions(tacfile), FILENAME_MAX-strlen(outfile)-1);
    if(verbose>2) printf("  %s\n", outfile);
    /* If this is not the first TAC, then copy it to the first place */
    if(i>0) {
      ret=tacCopyTacc(tac.c+i, tac.c, tac.sampleNr);
      if(ret!=0) {
        fprintf(stderr, "Error: cannot process TACs.\n");
        tacFree(&tac); return(9);
      }
    }
    /* If all samples in this TAC are missing (NaN), then save it only in force mode */
    if(force==0 && tacNotNaNs(&tac, -1)<1) continue;
    /* Save the TAC at the first position */
    if(verbose>2) printf("writing %s\n", outfile);
    fp=fopen(outfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing (%s)\n", outfile);
      tacFree(&tac); return(11);
    }
    ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(12);
    }
  }  // next TAC
  tacFree(&tac);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
