/** @file tacstudy.c
 *  @brief Set or get the study number in PET TAC and PAR files.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @todo Add support to parameter files, after parameter file i/o is ready.
 *  @test Test other TAC formats.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Set or get the valid TPC study number in regional TAC or SIF file.",
  "Valid TPC study number can not exceed 10 characters, and it must not",
  "contain spaces; it contains 1-5 letters (upper- or lowercase)",
  "followed by 1-5 digits.",
  " ",
  "If study number is not given as argument, or file does not contain it,",
  "it is automatically extracted from filename, when possible, by this and",
  "other applications.",
  " ",
  "Note that many file formats (including PMOD) do not support study numbers,",
  "but this application can still optionally add the study number as",
  "a comment, for example:",
  "# studynr := ab1006",
  " ",
  "Usage: @P [Options] filename [studynumber]",
  " ",
  "Options:",
  " -fn[=<N>]",
  "     Get study number from filename, optionally from the N first letters",
  "     (max 10) of file name, excluding possible path and extension.",
  " -df=<N>",
  "     Delete the first N letters from study number.",
  " -lower",
  "     Convert study number letters to lowercase.",
  " --force",
  "     Validity of new study number is not verified, and written in file",
  "     as comment line, if not otherwise supported. Maximum length for",
  "     study number is either 255 or 10 characters, depending on file format.",
  " -value",
  "     Print only the study number without file name.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacunit, tacadd, iftlist, iftedit, tacformat",
  " ",
  "Keywords: TAC, tool, simulation, study number",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int          ai, help=0, version=0, verbose=1;
  int          ret, toLower=0, force=0, print_value=0;
  unsigned int fromFilename=0, deleteFirst=0;
  char        *cptr, datfile[FILENAME_MAX], studynr[FILENAME_MAX];
  char         oldstudynr[FILENAME_MAX], tmp[FILENAME_MAX];
  TAC          tac;


  /*
   *  Get and check arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  datfile[0]=studynr[0]=oldstudynr[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "LOWER")==0) {
      toLower=1; continue;
    } else if(strncasecmp(cptr, "DF=", 3)==0) {
      cptr+=3; deleteFirst=atoi(cptr);
      if(deleteFirst>0 && deleteFirst<11) continue;
    } else if(strcasecmp(cptr, "FN")==0) {
      fromFilename=1; continue;
    } else if(strncasecmp(cptr, "FN=", 3)==0) {
      cptr+=3; fromFilename=atoi(cptr); 
      if(fromFilename>0) continue; // max length is check later
    } else if(strcasecmp(cptr, "FORCE")==0) {
      force=1; continue;
    } else if(strcasecmp(cptr, "VALUE")==0) {
      print_value=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!datfile[0]) {
      strlcpy(datfile, argv[ai], FILENAME_MAX); continue;
    } else if(!studynr[0]) {
      strlcpy(studynr, argv[ai], FILENAME_MAX);
      /* In case that this is filename, remove path and extension */
      filenameRmPath(studynr); filenameRmExtension(studynr);
      /* If anything is accepted, then be it */
      if(force==1 && strlen(studynr)>1) continue;
      /* If not forced, then check the validity as study number */
      ret=0;
      if(studynrVerify(studynr, 1)!=0) {
        /* Not valid; treat it like filename */
        ret=studynrFromFilename(argv[ai], studynr, force);
        if(fromFilename>1) studynr[fromFilename]=(char)0;
      }
      if(ret==0) continue;
      fprintf(stderr, "Error: invalid argument for study number '%s'\n", argv[ai]);
      return(1);
    }
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!datfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("datfile := %s\n", datfile);
    printf("studynr := %s\n", studynr);
    printf("force := %d\n", force);
    printf("toLower := %d\n", toLower);
    printf("fromFilename := %d\n", fromFilename);
    printf("deleteFirst := %d\n", deleteFirst);
    printf("print_value := %d\n", print_value);
  }

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;


  /* Check other arguments and options */
  if(fromFilename>1 && ((int)fromFilename-(int)deleteFirst)>10) {
    fprintf(stderr, "Error: too high value with option -fn.\n");
    return(1);
  }

  /* Take new study number from filename */
  if(!studynr[0] && fromFilename>0) {
    if(verbose>1) printf("getting new study number from filename\n");
    strcpy(tmp, datfile); filenameRmPath(tmp); filenameRmExtension(tmp);
    if(fromFilename>1) {
      tmp[fromFilename+deleteFirst]=(char)0;
      strcpy(studynr, tmp);
    } else {
      studynrFromFilename(tmp, studynr, force);
    }
    if(verbose>2) printf("studynr := %s\n", studynr);
    if(strlen(studynr)<1) {
      fprintf(stderr, "Error: cannot get study number from filename.\n");
      return(4);
    }
  }

  /* 
   *  Read datafile 
   */
  if(verbose>1) printf("reading %s\n", datfile);
  ret=tacRead(&tac, datfile, &status);
  if(ret) {
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), datfile);
    tacFree(&tac);
    return(2);
  }
  /* Get the current study number */
  ret=tacGetHeaderStudynr(&tac.h, oldstudynr, &status);
  /* Delete '.' from study number string */
  if(strcmp(oldstudynr, ".")==0) strcpy(oldstudynr, "");
  if(verbose>1) printf("infile.studynr := %s\n", oldstudynr);
  /* Convert to lowercase if necessary */
  if(toLower>0) {
    int i, len;
    len=strlen(oldstudynr);
    for(i=0; i<len; i++) {
      if(isdigit((int)oldstudynr[i])) continue;
      oldstudynr[i]=(char)tolower((int)oldstudynr[i]);
    }
  }
  /* Delete first letters if necessary */
  if(deleteFirst>0 && deleteFirst<strlen(oldstudynr)) {
    strcpy(tmp, oldstudynr+deleteFirst); strcpy(oldstudynr, tmp);
  }

  /* If new study number was not given then just print the current one */
  if(!studynr[0]) {
    strcpy(tmp, datfile); filenameRmPath(tmp);
    if(print_value) printf("%s\n", oldstudynr);
    else printf("%s := %s\n", tmp, oldstudynr);
    tacFree(&tac);
    return(0); /* and that's it */
  }

  /* Check the specified study number */
  if(studynr[0]) {
    if(verbose>2) printf("processing specified study number\n");

    /* Convert to lowercase if necessary */
    if(toLower>0) {
      if(verbose>2) printf("converting to lower case letters\n");
      //studynr_to_lowercase(studynr);
      int i, len;
      len=strlen(studynr);
      for(i=0; i<len; i++) {
        if(isdigit((int)studynr[i])) continue;
        studynr[i]=(char)tolower((int)studynr[i]);
      }
    }

    /* Delete first letters if necessary */
    if(deleteFirst>0) {
      if(verbose>2) printf("deleting first characters\n");
      if(studynr[0] && deleteFirst<strlen(studynr)) {
        strcpy(tmp, studynr+deleteFirst); strcpy(studynr, tmp);}
    }

    /* Check that it is not too long */
    if(strlen(studynr)>MAX_STUDYNR_LEN) {
      if(verbose>2) printf("new study number was too long, shortened\n");
      studynr[MAX_STUDYNR_LEN-1]=(char)0;
    }
    if(verbose>1) printf("studynr := %s\n", studynr);

    /* Test that it is still valid */
    if(force==0) {
      if(verbose>2) printf("verifying new study number because not forced\n");
      if(studynrVerify(studynr, 1)!=0) {
        fprintf(stderr, "Error: invalid study number.\n");
        if(verbose>2) printf("  studynr := '%s'\n", studynr);
        tacFree(&tac);
        return(1);
      }
    }
  }

  /* Change studynumber */
  if(verbose>0) 
    fprintf(stdout, "%s: '%s' -> '%s'\n", datfile, oldstudynr, studynr);
  ret=tacSetHeaderStudynr(&tac.h, studynr);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: cannot set new study number.\n");
    if(verbose>2) printf("  ret := %d\n", ret);
    tacFree(&tac);
    return(1);
  }
  /* and write the modified file */
  if(verbose>1) printf("writing %s\n", datfile);
  FILE *fp; fp=fopen(datfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", datfile);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, force, &status);
  fclose(fp);
  tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    return(12);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
