#!/bin/bash
#: Title      : test_taccat
#: Date       : 2024-06-18
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../taccat$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for  %s\n" $PROGRAM
printf "=====================================================================\n"


if [ ! -f test1a.tac ] || [ ! -f test1b.tac ] || [ ! -f correct_output1.tac ]; then
printf "start[minutes]	end[Bq/cc]	par__All	occ__All	tem__All	cer__All	weight\n" > test1a.tac
printf "0.00	1.00	2.410e+02	2.576e+02	2.181e+02	2.203e+02	1.863e+00\n" >> test1a.tac
printf "1.00	2.00	2.888e+03	2.558e+03	2.776e+03	3.518e+03	1.843e-01\n" >> test1a.tac
printf "2.00	6.00	7.605e+03	6.306e+03	6.520e+03	8.464e+03	7.599e-02\n" >> test1a.tac
printf "6.00	10.00	1.022e+04	9.182e+03	9.171e+03	1.112e+04	5.491e-02\n" >> test1a.tac

printf "start[minutes]	end[Bq/cc]	par__All	occ__All	tem__All	cer__All\n" > test1b.tac
printf "10.00	20.00	2.010e+04	1.176e+04	6.281e+03	2.293e+03\n" >> test1b.tac
printf "20.00	30.00	2.888e+04	1.555e+04	2.133e+03	8.551e+02\n" >> test1b.tac

printf "start[minutes]	end[Bq/cc]	par__All	occ__All	tem__All	cer__All\n" > correct_output1.tac
printf "0.00	1.00	2.410e+02	2.576e+02	2.181e+02	2.203e+02\n" >> correct_output1.tac
printf "1.00	2.00	2.888e+03	2.558e+03	2.776e+03	3.518e+03\n" >> correct_output1.tac
printf "2.00	6.00	7.605e+03	6.306e+03	6.520e+03	8.464e+03\n" >> correct_output1.tac
printf "6.00	10.00	1.022e+04	9.182e+03	9.171e+03	1.112e+04\n" >> correct_output1.tac
printf "10.00	20.00	2.010e+04	1.176e+04	6.281e+03	2.293e+03\n" >> correct_output1.tac
printf "20.00	30.00	2.888e+04	1.555e+04	2.133e+03	8.551e+02\n" >> correct_output1.tac
fi

if [ ! -f test1a.dft ] || [ ! -f test1b.dft ] || [ ! -f correct_output1.dft ]; then
  tacformat -f=DFT test1a.tac
  if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
  tacformat -f=DFT test1b.tac
  if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
  tacformat -f=DFT correct_output1.tac
  if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
fi

if [ ! -f correct_output1so.tac ]; then
printf "start[minutes]	end[Bq/cc]	par__All	occ__All	tem__All	cer__All\n" > correct_output1so.tac
printf "2.00	3.00	2.410e+02	2.576e+02	2.181e+02	2.203e+02\n" >> correct_output1so.tac
printf "3.00	4.00	2.888e+03	2.558e+03	2.776e+03	3.518e+03\n" >> correct_output1so.tac
printf "4.00	8.00	7.605e+03	6.306e+03	6.520e+03	8.464e+03\n" >> correct_output1so.tac
printf "8.00	12.00	1.022e+04	9.182e+03	9.171e+03	1.112e+04\n" >> correct_output1so.tac
printf "10.00	20.00	2.010e+04	1.176e+04	6.281e+03	2.293e+03\n" >> correct_output1so.tac
printf "20.00	30.00	2.888e+04	1.555e+04	2.133e+03	8.551e+02\n" >> correct_output1so.tac
fi

if [ ! -f correct_output1sof.tac ]; then
printf "start[minutes]	end[Bq/cc]	par__All	occ__All	tem__All	cer__All\n" > correct_output1sof.tac
printf "2.00	3.00	2.410e+02	2.576e+02	2.181e+02	2.203e+02\n" >> correct_output1sof.tac
printf "3.00	4.00	2.888e+03	2.558e+03	2.776e+03	3.518e+03\n" >> correct_output1sof.tac
printf "4.00	8.00	7.605e+03	6.306e+03	6.520e+03	8.464e+03\n" >> correct_output1sof.tac
printf "8.00	12.00	1.022e+04	9.182e+03	9.171e+03	1.112e+04\n" >> correct_output1sof.tac
printf "20.00	30.00	2.888e+04	1.555e+04	2.133e+03	8.551e+02\n" >> correct_output1sof.tac
fi

if [ ! -f correct_output1sos.tac ]; then
printf "start[minutes]	end[Bq/cc]	par__All	occ__All	tem__All	cer__All\n" > correct_output1sos.tac
printf "2.00	3.00	2.410e+02	2.576e+02	2.181e+02	2.203e+02\n" >> correct_output1sos.tac
printf "3.00	4.00	2.888e+03	2.558e+03	2.776e+03	3.518e+03\n" >> correct_output1sos.tac
printf "4.00	8.00	7.605e+03	6.306e+03	6.520e+03	8.464e+03\n" >> correct_output1sos.tac
printf "10.00	20.00	2.010e+04	1.176e+04	6.281e+03	2.293e+03\n" >> correct_output1sos.tac
printf "20.00	30.00	2.888e+04	1.555e+04	2.133e+03	8.551e+02\n" >> correct_output1sos.tac
fi



if [ ! -f test2a.dat ] || [ ! -f test2b.dat ]; then
printf "# Comment in test2a.dat\n" > test2a.dat
printf "# injection_time := 2006-01-09 16:49:03\n" >> test2a.dat
printf "0 0\n" >> test2a.dat
printf "1 -1\n" >> test2a.dat
printf "2 2\n" >> test2a.dat
printf "3 0\n" >> test2a.dat
printf "4 4\n" >> test2a.dat
printf "5 12\n" >> test2a.dat
printf "6 550\n" >> test2a.dat
printf "7 1009\n" >> test2a.dat
printf "8 1888\n" >> test2a.dat
printf "9 1934\n" >> test2a.dat
printf "10 1878\n" >> test2a.dat
printf "11 1789\n" >> test2a.dat
printf "12 1604\n" >> test2a.dat
printf "13 1557\n" >> test2a.dat
printf "14 1445\n" >> test2a.dat
printf "15 1398\n" >> test2a.dat
printf "16 1313\n" >> test2a.dat
printf "17 1201\n" >> test2a.dat
printf "18 1189\n" >> test2a.dat
printf "19 888\n" >> test2a.dat
printf "20 765\n" >> test2a.dat
printf "21 0\n" >> test2a.dat
printf "22 2\n" >> test2a.dat
printf "23 -1\n" >> test2a.dat
printf "24 7\n" >> test2a.dat

printf "# Comment in test2b.dat\n" > test2b.dat
printf "# unit := kBq/ml\n" >> test2b.dat
printf "# isotope := C-11\n" >> test2b.dat
printf "20.9   655\n" >> test2b.dat
printf "35.1   233\n" >> test2b.dat
printf "47.2   187\n" >> test2b.dat
printf "60.6   165\n" >> test2b.dat
fi

if [ ! -f correct_output2a.dat ]; then
printf "0 0\n" > correct_output2a.dat
printf "1 -1\n" >> correct_output2a.dat
printf "2 2\n" >> correct_output2a.dat
printf "3 0\n" >> correct_output2a.dat
printf "4 4\n" >> correct_output2a.dat
printf "5 12\n" >> correct_output2a.dat
printf "6 550\n" >> correct_output2a.dat
printf "7 1009\n" >> correct_output2a.dat
printf "8 1888\n" >> correct_output2a.dat
printf "9 1934\n" >> correct_output2a.dat
printf "10 1878\n" >> correct_output2a.dat
printf "11 1789\n" >> correct_output2a.dat
printf "12 1604\n" >> correct_output2a.dat
printf "13 1557\n" >> correct_output2a.dat
printf "14 1445\n" >> correct_output2a.dat
printf "15 1398\n" >> correct_output2a.dat
printf "16 1313\n" >> correct_output2a.dat
printf "17 1201\n" >> correct_output2a.dat
printf "18 1189\n" >> correct_output2a.dat
printf "19 888\n" >> correct_output2a.dat
printf "20 765\n" >> correct_output2a.dat
printf "20.9 655\n" >> correct_output2a.dat
printf "21 0\n" >> correct_output2a.dat
printf "22 2\n" >> correct_output2a.dat
printf "23 -1\n" >> correct_output2a.dat
printf "24 7\n" >> correct_output2a.dat
printf "35.1 233\n" >> correct_output2a.dat
printf "47.2 187\n" >> correct_output2a.dat
printf "60.6 165\n" >> correct_output2a.dat
printf "# Comment in test2a.dat\n" >> correct_output2a.dat
printf "# injection_time := 2006-01-09 16:49:03\n" >> correct_output2a.dat
printf "# unit := kBq/ml\n" >> correct_output2a.dat
printf "# isotope := C-11\n" >> correct_output2a.dat
fi


if [ ! -f correct_output2f.dat ]; then
printf "0 0\n" > correct_output2f.dat
printf "1 -1\n" >> correct_output2f.dat
printf "2 2\n" >> correct_output2f.dat
printf "3 0\n" >> correct_output2f.dat
printf "4 4\n" >> correct_output2f.dat
printf "5 12\n" >> correct_output2f.dat
printf "6 550\n" >> correct_output2f.dat
printf "7 1009\n" >> correct_output2f.dat
printf "8 1888\n" >> correct_output2f.dat
printf "9 1934\n" >> correct_output2f.dat
printf "10 1878\n" >> correct_output2f.dat
printf "11 1789\n" >> correct_output2f.dat
printf "12 1604\n" >> correct_output2f.dat
printf "13 1557\n" >> correct_output2f.dat
printf "14 1445\n" >> correct_output2f.dat
printf "15 1398\n" >> correct_output2f.dat
printf "16 1313\n" >> correct_output2f.dat
printf "17 1201\n" >> correct_output2f.dat
printf "18 1189\n" >> correct_output2f.dat
printf "19 888\n" >> correct_output2f.dat
printf "20 765\n" >> correct_output2f.dat
printf "21 0\n" >> correct_output2f.dat
printf "22 2\n" >> correct_output2f.dat
printf "23 -1\n" >> correct_output2f.dat
printf "24 7\n" >> correct_output2f.dat
printf "35.1 233\n" >> correct_output2f.dat
printf "47.2 187\n" >> correct_output2f.dat
printf "60.6 165\n" >> correct_output2f.dat
printf "# Comment in test2a.dat\n" >> correct_output2f.dat
printf "# injection_time := 2006-01-09 16:49:03\n" >> correct_output2f.dat
printf "# unit := kBq/ml\n" >> correct_output2f.dat
printf "# isotope := C-11\n" >> correct_output2f.dat
fi


if [ ! -f correct_output2s.dat ]; then
printf "0 0\n" > correct_output2s.dat
printf "1 -1\n" >> correct_output2s.dat
printf "2 2\n" >> correct_output2s.dat
printf "3 0\n" >> correct_output2s.dat
printf "4 4\n" >> correct_output2s.dat
printf "5 12\n" >> correct_output2s.dat
printf "6 550\n" >> correct_output2s.dat
printf "7 1009\n" >> correct_output2s.dat
printf "8 1888\n" >> correct_output2s.dat
printf "9 1934\n" >> correct_output2s.dat
printf "10 1878\n" >> correct_output2s.dat
printf "11 1789\n" >> correct_output2s.dat
printf "12 1604\n" >> correct_output2s.dat
printf "13 1557\n" >> correct_output2s.dat
printf "14 1445\n" >> correct_output2s.dat
printf "15 1398\n" >> correct_output2s.dat
printf "16 1313\n" >> correct_output2s.dat
printf "17 1201\n" >> correct_output2s.dat
printf "18 1189\n" >> correct_output2s.dat
printf "19 888\n" >> correct_output2s.dat
printf "20 765\n" >> correct_output2s.dat
printf "20.9 655\n" >> correct_output2s.dat
printf "35.1 233\n" >> correct_output2s.dat
printf "47.2 187\n" >> correct_output2s.dat
printf "60.6 165\n" >> correct_output2s.dat
printf "# Comment in test2a.dat\n" >> correct_output2s.dat
printf "# injection_time := 2006-01-09 16:49:03\n" >> correct_output2s.dat
printf "# unit := kBq/ml\n" >> correct_output2s.dat
printf "# isotope := C-11\n" >> correct_output2s.dat
fi

if [ ! -f correct_output2c.dat ]; then
printf "0 0\n" > correct_output2c.dat
printf "1 -1\n" >> correct_output2c.dat
printf "2 2\n" >> correct_output2c.dat
printf "3 0\n" >> correct_output2c.dat
printf "4 4\n" >> correct_output2c.dat
printf "5 12\n" >> correct_output2c.dat
printf "20.9 655\n" >> correct_output2c.dat
printf "35.1 233\n" >> correct_output2c.dat
printf "47.2 187\n" >> correct_output2c.dat
printf "60.6 165\n" >> correct_output2c.dat
printf "# Comment in test2a.dat\n" >> correct_output2c.dat
printf "# injection_time := 2006-01-09 16:49:03\n" >> correct_output2c.dat
printf "# unit := kBq/ml\n" >> correct_output2c.dat
printf "# isotope := C-11\n" >> correct_output2c.dat
fi


if [ ! -f test3a.dat ] || [ ! -f test3b.dat ]; then
printf "0.0 0\n" >  test3a.dat
printf "0.1 0\n" >> test3a.dat
printf "0.2 8\n" >> test3a.dat
printf "0.3 9\n" >> test3a.dat
printf "0.4 8\n" >> test3a.dat
printf "0.5 6\n" >> test3a.dat
printf "0.6 4\n" >> test3a.dat
printf "0.7 3\n" >> test3a.dat
printf "0.8 3\n" >> test3a.dat
printf "0.9 2\n" >> test3a.dat
printf "1.0 2\n" >> test3a.dat

printf "1.0 1\n" >  test3b.dat
printf "2.0 1\n" >> test3b.dat
printf "3.0 0\n" >> test3b.dat
printf "4.0 0\n" >> test3b.dat
printf "5.0 0\n" >> test3b.dat
fi

if [ ! -f correct_output3a.dat ]; then
printf "0.0 0\n" >  correct_output3a.dat
printf "0.1 0\n" >> correct_output3a.dat
printf "0.2 8\n" >> correct_output3a.dat
printf "0.3 9\n" >> correct_output3a.dat
printf "0.4 8\n" >> correct_output3a.dat
printf "0.5 6\n" >> correct_output3a.dat
printf "0.6 4\n" >> correct_output3a.dat
printf "0.7 3\n" >> correct_output3a.dat
printf "0.8 3\n" >> correct_output3a.dat
printf "0.9 2\n" >> correct_output3a.dat
printf "1.0 2\n" >> correct_output3a.dat
printf "2.0 1\n" >> correct_output3a.dat
printf "3.0 0\n" >> correct_output3a.dat
printf "4.0 0\n" >> correct_output3a.dat
printf "5.0 0\n" >> correct_output3a.dat
fi

if [ ! -f correct_output3b.dat ]; then
printf "0.0 0\n" >  correct_output3b.dat
printf "0.1 0\n" >> correct_output3b.dat
printf "0.2 8\n" >> correct_output3b.dat
printf "0.3 9\n" >> correct_output3b.dat
printf "0.4 8\n" >> correct_output3b.dat
printf "0.5 6\n" >> correct_output3b.dat
printf "0.6 4\n" >> correct_output3b.dat
printf "0.7 3\n" >> correct_output3b.dat
printf "0.8 3\n" >> correct_output3b.dat
printf "0.9 2\n" >> correct_output3b.dat
printf "1.0 1\n" >> correct_output3b.dat
printf "2.0 1\n" >> correct_output3b.dat
printf "3.0 0\n" >> correct_output3b.dat
printf "4.0 0\n" >> correct_output3b.dat
printf "5.0 0\n" >> correct_output3b.dat
fi

if [ ! -f correct_output3c.dat ]; then
printf "0.0 0\n" >  correct_output3c.dat
printf "0.1 0\n" >> correct_output3c.dat
printf "0.2 8\n" >> correct_output3c.dat
printf "0.3 9\n" >> correct_output3c.dat
printf "0.4 8\n" >> correct_output3c.dat
printf "0.5 6\n" >> correct_output3c.dat
printf "0.6 4\n" >> correct_output3c.dat
printf "0.7 3\n" >> correct_output3c.dat
printf "0.8 3\n" >> correct_output3c.dat
printf "0.9 2\n" >> correct_output3c.dat
printf "1.0 2\n" >> correct_output3c.dat
printf "1.0 1\n" >> correct_output3c.dat
printf "2.0 1\n" >> correct_output3c.dat
printf "3.0 0\n" >> correct_output3c.dat
printf "4.0 0\n" >> correct_output3c.dat
printf "5.0 0\n" >> correct_output3c.dat
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case 1: Reasonable data in PMOD TTAC format without options. \n"
printf " Test case 2: New file name given. \n"
printf " Test case 3: First file contains weights. \n"
printf " Expected result: Correctly catenated data is written in new file. \n"
printf " Expected result 2: Weights are silently ignored. \n"

rm -f output.tac
$PROGRAM test1a.tac test1b.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 -xunit=y -yunit=y -tacnames=y correct_output1.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacweigh -list output.tac
if [ $? -eq 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case: Output file exists. \n"
printf " Test case 2: Option -both. \n"
printf " Expected result: Existing file is overwritten with new file. \n"

cp -f test1a.tac output.tac
$PROGRAM -both test1a.tac test1b.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct_output1.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.2 \n"
printf " Test case 1: Output file name is NOT given. \n"
printf " Expected result: First file is overwritten with catenated data. \n"

cp -f test1a.tac output.tac
$PROGRAM output.tac test1b.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct_output1.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.3 \n"
printf " Test case: Still no overlap in files, but option -first or -second is given. \n"
printf " Expected result: Correctly catenated data is written in new file. \n"

rm -f output.tac
$PROGRAM -first test1a.tac test1b.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 -xunit=y -yunit=y -tacnames=y correct_output1.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.tac
$PROGRAM -second test1a.tac test1b.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 -xunit=y -yunit=y -tacnames=y correct_output1.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.tac
$PROGRAM -cut=10 test1a.tac test1b.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 -xunit=y -yunit=y -tacnames=y correct_output1.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case: DFT TTAC format. \n"
printf " Expected result: Correctly catenated data is written in new file. \n\n"

rm -f output.dft
$PROGRAM test1a.dft test1b.dft output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 -xunit=y -yunit=y -tacnames=y correct_output1.dft output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.0 \n"
printf " Test case: Slightly overlapping PMOD TTAC data. \n"
printf " Test case 2: Option -BOTH. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

rm -f output.tac
tactime test1a.tac +2 temp1a.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -BOTH temp1a.tac test1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times is ignored."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct_output1so.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.1 \n"
printf " Test case: Same as before but without option -BOTH. \n"
printf " Expected result: Error. \n\n"

rm -f output.tac
$PROGRAM temp1a.tac test1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "overlapping sample times."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.2 \n"
printf " Test case: Same as before but with option -FIRST. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

rm -f output.tac
$PROGRAM -FIRST temp1a.tac test1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times will be removed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct_output1sof.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.3 \n"
printf " Test case: Same as before but with option -SECOND. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

rm -f output.tac
$PROGRAM -SECOND temp1a.tac test1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times will be removed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct_output1sos.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.4 \n"
printf " Test case: Same as before but with option -CUT at end of first data. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

rm -f output.tac
$PROGRAM -CUT=12 temp1a.tac test1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times will be removed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct_output1sof.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.5 \n"
printf " Test case: Same as before but with option -CUT at start of second data. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

rm -f output.tac
$PROGRAM -CUT=10 temp1a.tac test1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times will be removed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct_output1sos.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n ----------------------------------------------------------------\n"

printf "\n 1.3.0 \n"
printf " Test case: Files have different units. \n"
printf " Expected result: Second dataset is converted to units of first dataset. \n\n"

cp -f test1b.tac temp1b.tac
rm -f output.tac
tacunit -xconv=min -yconv=kBq/cc temp1b.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test1a.tac temp1b.tac output.tac 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct_output1.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n ----------------------------------------------------------------\n"

printf "\n 2.0.0 \n"
printf " Test case: Overlapping TAC data in simple text format, units missing from the first file. \n"
printf " Test case 2: No options. \n"
printf " Expected result: Error about overlap or missing units. \n\n"

rm -f output.dat
$PROGRAM test2a.dat test2b.dat output.dat  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "unknown time units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "unknown concentration units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.1 \n"
printf " Test case: Overlapping TAC data in simple text format, units missing from the first file. \n"
printf " Test case 2: Option -force. \n"
printf " Expected result: Error. \n\n"

rm -f output.dat
$PROGRAM -force test2a.dat test2b.dat output.dat  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "overlapping sample times."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.2 \n"
printf " Test case: Overlapping TAC data in simple text format, units missing from the first file. \n"
printf " Test case 2: Option -both. \n"
printf " Expected result: Warning on missing units. \n"
printf " Expected result 2: Error on non-compatible units. \n\n"

rm -f output.dat
$PROGRAM -both test2a.dat test2b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "unknown time units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "unknown concentration units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "non-compatible TAC concentration units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case: Overlapping TAC data in simple text format. \n"
printf " Test case 2: Options -both and -force. \n"
printf " Test case 3: Units and other information as comments in either file. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n"
printf " Expected result 3: Important comments from both files retained. \n\n"

rm -f output.dat
$PROGRAM -both -force test2a.dat test2b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times is ignored."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=0.1 -xunit=y -yunit=y correct_output2a.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

iftmatch test2a.dat output.dat injection_time
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch test2b.dat output.dat unit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch test2b.dat output.dat isotope
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.2.0 \n"
printf " Test case: Overlapping TAC data in simple text format. \n"
printf " Test case 2: Option -first and -force. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

rm -f output.dat
$PROGRAM -first -force test2a.dat test2b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times will be removed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=0.1 -xunit=y -yunit=y correct_output2f.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.1 \n"
printf " Test case: Overlapping TAC data in simple text format. \n"
printf " Test case 2: Option -second and -force. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

rm -f output.dat
$PROGRAM -second -force test2a.dat test2b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times will be removed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=0.1 -xunit=y -yunit=y correct_output2s.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.3.0 \n"
printf " Test case: Overlapping TAC data in simple text format. \n"
printf " Test case 2: Option -cut setting time to end of first data. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

rm -f output.dat
$PROGRAM -cut=24 -force test2a.dat test2b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times will be removed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=0.1 -xunit=y -yunit=y correct_output2f.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.1 \n"
printf " Test case: Overlapping TAC data in simple text format. \n"
printf " Test case 2: Option -cut setting time to start of second data. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

rm -f output.dat
$PROGRAM -cut=20.8 -force test2a.dat test2b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times will be removed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=0.1 -xunit=y -yunit=y correct_output2s.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.2 \n"
printf " Test case: Overlapping TAC data in simple text format. \n"
printf " Test case 2: Option -cut setting time well inside first data. \n"
printf " Test case 3: Cut time given in file. \n"
printf " Expected result: Warning of overlap. \n"
printf " Expected result 2: Correctly catenated data is written in new file. \n\n"

printf "cuttime\nx := 5.5\n" > cuttime.ift
rm -f output.tac

$PROGRAM -cut=cuttime.ift -force test2a.dat test2b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "overlap in sample times will be removed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=0.1 -xunit=y -yunit=y correct_output2c.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.4.0 \n"
printf " Test case: Just one sample overlapping in simple text format. \n"
printf " Test case 2: Options -first, -second, and -both. \n"
printf " Expected result: Correctly catenated data is written in new file. \n\n"

$PROGRAM test3a.dat test3b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "overlapping sample times."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "Option -first\n"
$PROGRAM -first test3a.dat test3b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=n -yunit=n correct_output3a.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "Option -second\n"
$PROGRAM -second test3a.dat test3b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=n -yunit=n correct_output3b.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "Option -both\n"
$PROGRAM -both test3a.dat test3b.dat output.dat 1>stdout.txt 2>&1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -xunit=n -yunit=n correct_output3c.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "=====================================================================\n"
printf "  More error handling\n"
printf "=====================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Appropriate error. \n\n"

$PROGRAM test1a.tac test1b.tac output.tac extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM test1a.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: File does not exist. \n"
printf " Expected result: Appropriate error. \n\n"

$PROGRAM nonexisting.tac test1b.tac output.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM test1a.tac nonexisting.tac output.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf " First file is not overwritten in case of missing second file. \n"

tacadd -ovr temp1a.tac test1a.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM temp1a.tac nonexisting.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch test1a.tac temp1a.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.2.0 \n"
printf " Test case: Files contain different number of TACs. \n"
printf " Expected result: Appropriate error. \n\n"

tacadd -ovr temp1a.tac test1a.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacdel temp1a.tac cer
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp1b.tac test1b.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacdel temp1b.tac cer
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f output.tac

$PROGRAM temp1a.tac test1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "different number of TACs."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.tac

$PROGRAM test1a.tac temp1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "different number of TACs."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.1 \n"
printf " Test case: Files contain same number of TACs but different names. \n"
printf " Expected result: Appropriate error. \n\n"

tacadd -ovr temp1a.tac test1a.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacren temp1a.tac cer Else
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp1b.tac test1b.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacren temp1b.tac cer Else
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f output.tac

$PROGRAM temp1a.tac test1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "TAC names do not match."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.tac

$PROGRAM test1a.tac temp1b.tac output.tac 1>stdout.txt 2>&1 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "TAC names do not match."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"






printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0
