#!/bin/bash
#: Title      : test_tacunit
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../tacunit$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for  %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n verifying that required data exists \n"

if [ ! -f tacunit_correct2.dft ] || [ ! -f tacunit_correct2.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f tacunit_test.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f tacunit_test1.dat ] || [ ! -f tacunit_test1.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f tacunit_test2.dat ] || [ ! -f tacunit_test2.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case: Nonexisting file. \n"
printf " Expected result: Error. \n"

$PROGRAM nonexistingfile.tac 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot open file"
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case: Binary file as data file should not crash program. \n"
printf " Expected result: Error. \n"

$PROGRAM -d $PROGRAM 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid file format"
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n-------------------------------------------------------------------\n"


printf "\n 2.0.0 \n"
printf " Test case 1: DAT format. \n"
printf " Test case 2: Listing units\n"
printf " Expected result: Units correctly listed. \n"

cp -f tacunit_test1.dat test.dat

$PROGRAM test.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt timeunit unknown
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt unit unknown
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -x=y -y=y -xunit=y -yunit=y -abs=0.00001 test.dat tacunit_test1.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 2.1.0 \n"
printf " Test case 1: DAT format. \n"
printf " Test case 2: Setting units\n"
printf " Expected result 1: Units correctly set. \n"
printf " Expected result 2: Values not changed. \n\n"


$PROGRAM -xset=min -yset=Bq/cc test.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval test.dat timeunit min
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval test.dat unit Bq/ml
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -x=y -y=y -xunit=n -yunit=n -abs=0.00001 test.dat tacunit_test1.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"

$PROGRAM test.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt timeunit min
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt unit Bq/ml
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "passed.\n\n"



printf "\n 2.2.0 \n"
printf " Test case 1: DAT format. \n"
printf " Test case 2: Converting units\n"
printf " Expected result 1: Units correctly set. \n"
printf " Expected result 2: Values changed. \n\n"

cp -f tacunit_test2.dat test.dat

$PROGRAM -xconv=sec -yconv=kBq/cc test.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"

tacmatch -x=y -y=y -xunit=y -yunit=y -abs=0.00001 test.dat tacunit_correct2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"


printf "\n-------------------------------------------------------------------\n"


printf "\n 3.0.0 \n"
printf " Test case 1: DFT format. \n"
printf " Test case 2: Listing units\n"
printf " Expected result: Units correctly listed. \n"

cp -f tacunit_test1.dft test.dft

$PROGRAM test.dft 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt timeunit unknown
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt unit unknown
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -x=y -y=y -xunit=y -yunit=y -abs=0.00001 test.dft tacunit_test1.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 3.1.0 \n"
printf " Test case 1: DFT format. \n"
printf " Test case 2: Setting units\n"
printf " Expected result 1: Units correctly set. \n"
printf " Expected result 2: Values not changed. \n\n"

$PROGRAM -xset=min -yset=Bq/cc test.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.dft 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt timeunit min
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt unit Bq/ml
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "passed.\n\n"

tacmatch -x=y -y=y -xunit=n -yunit=n -abs=0.00001 test.dft tacunit_test1.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"




printf "\n 3.2.0 \n"
printf " Test case 1: DFT format. \n"
printf " Test case 2: Converting units\n"
printf " Expected result 1: Units correctly set. \n"
printf " Expected result 2: Values changed. \n\n"

cp -f tacunit_test2.dft test.dft

$PROGRAM -xconv=sec -yconv=kBq/cc test.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"

tacmatch -x=y -y=y -xunit=y -yunit=y -abs=0.00001 test.dft tacunit_correct2.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"



printf "\n 3.3.0 \n"
printf " Test case 1: mL versus gram. \n"
printf " Test case 2: Converting not supported when density not given.\n"
printf " Expected result: Error. \n\n"

cp -f tacunit_test2.dft test.dft

$PROGRAM -xconv=sec -yconv=kBq/g test.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "conversion from 'Bq/mL' to 'kBq/g' not supported."
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 3.3.1 \n"
printf " Test case 1: mL versus gram. \n"
printf " Test case 2: Converting supported when density is given.\n"
printf " Expected result 1: Units correctly set. \n"
printf " Expected result 2: Values changed. \n\n"

$PROGRAM -d3 -density=1.1 -xconv=sec -yconv=kBq/g test.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.dft 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt timeunit sec
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt unit kBq/g
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "passed.\n\n"

$PROGRAM -density=1.1 -xconv=min -yconv=Bq/cc test.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"

tacmatch -x=y -y=y -xunit=y -yunit=y -abs=0.00001 test.dft tacunit_test2.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"

printf "\n-------------------------------------------------------------------\n"

printf "\n 4.0.0 \n"
printf " Test case: Unit can be copied from one file to another in 2 steps. \n"
printf " Expected result: Units correctly set. \n\n"

cp -f tacunit_test1.dft test1.dft
cp -f tacunit_test2.dft test2.dft
rm -f stdout.txt

xunit=$( $PROGRAM -x test2.dft )
yunit=$( $PROGRAM -y test2.dft )

$PROGRAM -xset=$xunit -yset=$yunit test1.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test1.dft > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt timeunit min
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt unit Bq/ml
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"



printf "=====================================================================\n"
printf "  All passed!\n"
printf "=====================================================================\n"
exit 0
